package ga.view.appstate.menu;

import ga.view.appstate.SceneState;
import ga.view.config.GASettings;
import ga.view.config.ViewSettings;
import ga.view.interfaces.MouseListener;

import java.util.ArrayList;
import java.util.List;
import java.util.logging.Logger;

import com.jme3.app.Application;
import com.jme3.app.state.AppStateManager;
import com.jme3.collision.CollisionResult;
import com.jme3.collision.CollisionResults;
import com.jme3.font.BitmapFont;
import com.jme3.font.BitmapFont.Align;
import com.jme3.font.BitmapText;
import com.jme3.font.Rectangle;
import com.jme3.input.MouseInput;
import com.jme3.input.controls.MouseButtonTrigger;
import com.jme3.material.Material;
import com.jme3.math.ColorRGBA;
import com.jme3.math.FastMath;
import com.jme3.math.Ray;
import com.jme3.math.Vector3f;
import com.jme3.renderer.queue.RenderQueue.ShadowMode;
import com.jme3.scene.Geometry;
import com.jme3.scene.Node;
import com.jme3.scene.shape.Box;

/**
 * App state for the main menu.
 * 
 * @since 12.08.2012
 * @author Stephan Dreyer
 */
public class MenuState extends SceneState {

  // the logger for this class
  private static final Logger LOGGER = Logger.getLogger(MenuState.class
      .getName());

  private MenuMouseListener doubleClickListener;

  private final MenuListener listener;
  private final List<MenuNode> menuNodes = new ArrayList<MenuState.MenuNode>();

  private final ViewSettings viewSettings;

  /**
   * Instantiates a new menu state.
   * 
   * @param listener
   *          the listener
   * @param viewSettings
   *          the view settings
   */
  public MenuState(final MenuListener listener, final ViewSettings viewSettings) {
    this.listener = listener;
    this.viewSettings = viewSettings;
  }

  @Override
  public void initialize(final AppStateManager stateManager,
      final Application app) {
    super.initialize(stateManager, app);

    cam.setLocation(new Vector3f(0f, 4.5f, 0f));
    cam.lookAt(Vector3f.ZERO, Vector3f.UNIT_Y);
    // cam.setParallelProjection(true);

    final float frustumSize = 2f;
    final float aspect = (float) cam.getWidth() / cam.getHeight();
    // cam.setFrustum(-1000, 1000, -aspect * frustumSize, aspect * frustumSize,
    // frustumSize, -frustumSize);

    inputManager.addMapping("choose", new MouseButtonTrigger(
        MouseInput.BUTTON_LEFT));

    doubleClickListener = new MenuMouseListener();
    inputManager.addListener(doubleClickListener, "choose");

    // final ScreenshotAppState screenShotState = new ScreenshotAppState(".");
    // this.stateManager.attach(screenShotState);

    float x = 2f;
    float z = 1f;

    for (final GASettings gaSetting : viewSettings) {
      final MenuNode node = new MenuNode(gaSetting);
      node.setOrgLocation(new Vector3f(x, 0f, z));

      menuNodes.add(node);
      rootNode.attachChild(node);

      x -= 2f;
      if (x < -2f) {
        x = 2f;
        z = -1f;
      }
    }
  }

  /**
   * Finds spatials in the scene that has been clicked.
   * 
   * @param node
   *          The parent node to check for clicks.
   * @return The results of the click.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  private CollisionResults findPick(final Node node) {
    final Vector3f origin = cam.getWorldCoordinates(
        inputManager.getCursorPosition(), 0.0f);
    final Vector3f direction = cam.getWorldCoordinates(
        inputManager.getCursorPosition(), 0.3f);
    direction.subtractLocal(origin).normalizeLocal();

    final Ray ray = new Ray(origin, direction);
    final CollisionResults results = new CollisionResults();
    node.collideWith(ray, results);
    return results;
  }

  @Override
  public void setEnabled(final boolean enabled) {
    super.setEnabled(enabled);

    if (doubleClickListener != null) {
      doubleClickListener.setEnabled(enabled);
    }
  }

  /**
   * A node for one model in the menu. Displays the image of the model.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  private class MenuNode extends Node {
    private Vector3f targetLocation;
    private Vector3f orgLocation;

    private boolean selected;
    private final GASettings gaSetting;
    private final String name;

    private final BitmapText text;

    /**
     * Instantiates a new menu node.
     * 
     * @param gaSetting
     *          the ga setting
     * 
     * @since 12.08.2012
     * @author Stephan Dreyer
     */
    public MenuNode(final GASettings gaSetting) {
      this.gaSetting = gaSetting;
      this.name = gaSetting.get("name", "name");
      setName(name + " node");

      final Material whitemat = new Material(assetManager,
          "Common/MatDefs/Misc/Unshaded.j3md");
      whitemat.setColor("Color", ColorRGBA.White);

      final float boxSizeX = .8f;
      final float boxSizeY = .6f;
      final float borderSize = .05f;

      final Material mat = new Material(assetManager,
          "Common/MatDefs/Misc/Unshaded.j3md");
      mat.setTexture("ColorMap",
          assetManager.loadTexture(gaSetting.get("iconImage", "")));

      final Geometry geo = new Geometry("Menu node", new Box(boxSizeY, 0.01f,
          boxSizeX));
      geo.setMaterial(mat);
      geo.setShadowMode(ShadowMode.Off);

      final Box box = new Box(boxSizeY + borderSize, 0f, boxSizeX + borderSize);
      final Geometry outterBox = new Geometry(name + " outter box", box);
      outterBox.setMaterial(whitemat);

      attachChild(outterBox);

      attachChild(geo);

      final BitmapFont font = assetManager
          .loadFont("Interface/Fonts/Default.fnt");

      text = new BitmapText(font, false);
      text.setName(name + " text");
      text.setSize(0.1f);
      text.setText(name);
      text.setCullHint(CullHint.Never);

      text.setBox(new Rectangle(-1f, 0f, 2f, text.getLineHeight()));
      text.setAlignment(Align.Center);

      text.setLocalTranslation(boxSizeY + borderSize, 0f, 0f);

      rotate(0f, FastMath.DEG_TO_RAD * 90f, 0f);

      attachChild(text);
      text.rotate(FastMath.DEG_TO_RAD * 270f, FastMath.DEG_TO_RAD * 90f, 0);

    }

    /**
     * Sets the original location for this node.
     * 
     * @param orgLocation
     *          Original location.
     * 
     * @since 12.08.2012
     * @author Stephan Dreyer
     */
    public void setOrgLocation(final Vector3f orgLocation) {
      this.orgLocation = orgLocation;
      setLocalTranslation(orgLocation);
    }

    @Override
    public void updateLogicalState(final float tpf) {
      super.updateLogicalState(tpf);

      if (targetLocation != null) {
        final Vector3f loc = getLocalTranslation();

        final float distance = targetLocation.distance(loc);

        if (distance > .05f) {
          // the move step should be limited by the distance to the target
          final float maxMoveStep = Math.min(distance, tpf * 3f);

          // calc the delta translation by subtracting the direction vector
          // multiplied with the step size from the target location
          final Vector3f dir = targetLocation.subtract(loc).normalize()
              .mult(maxMoveStep);
          setLocalTranslation(loc.add(dir));
        }
      }
    }

    /**
     * Sets the node selected or deselected. If selected, it moves nearer to the
     * camera.
     * 
     * @param select
     *          Selected.
     * 
     * @since 12.08.2012
     * @author Stephan Dreyer
     */
    public void setSelected(final boolean select) {
      if (select != selected) {
        selected = select;
        LOGGER.info(this + " is selected: " + select);
        if (select) {
          this.targetLocation = orgLocation.mult(.9f).add(0f, .5f, 0f);
        } else {
          this.targetLocation = this.orgLocation;
        }
      } else if (selected) {
        doubleClickListener.setEnabled(false);
        listener.modelSelected(name);

        // / reset
        this.targetLocation = this.orgLocation;
        selected = false;
      }
    }
  }

  /**
   * Mouse listener that handles clicks on the menu elements.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  private class MenuMouseListener extends MouseListener {
    @Override
    public void onAction(final String name, final boolean keyPressed,
        final boolean isDoubleClick, final float tpf) {
      if (!keyPressed) {
        final CollisionResults results = findPick(rootNode);

        MenuNode selectedNode = null;

        final CollisionResult r = results.getClosestCollision();
        if (r != null && r.getGeometry() != null
            && r.getGeometry().getParent() != null
            && r.getGeometry().getParent() instanceof MenuNode) {
          selectedNode = (MenuNode) r.getGeometry().getParent();
        }

        for (final MenuNode node : menuNodes) {
          node.setSelected(node == selectedNode);
        }
      }
    }
  }
}
